<?php

namespace App\Modules\Blog\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\Blog\Http\Resources\PostResource;
use App\Modules\Blog\Models\Category;
use App\Modules\Blog\Models\Post;
use Carbon\Carbon;
use Illuminate\Http\Request;


class PostsController extends Controller
{
    public function createPost() : \Illuminate\Http\JsonResponse
    {
        $post = Post::create([
            'status' => 'DRAFT',
            'slug' => Carbon::now()->timestamp,
            'category_id' => NULL,
            'content_body' => [
                'en' => [
                    'preview_title' => ' ',
                    'preview_body' => ' ',
                    'preview_image' => ' ',
                    'body' => [],
                ],
                'ru' => [
                    'preview_title' => ' ',
                    'preview_body' => ' ',
                    'preview_image' => ' ',
                    'body' => [],
                ],
            ],
        ]);

        $post->slug = $post->id;
        $post->save();
        $template = ['en' => [], 'ru' => []];
        return response()->success('Post created successful', [
            'post' => [
                'id' => $post->id,
                'status' => $post->status,
                'category' => $template,
                'tags' => $template,
                'created_at' => $post->created_at,
                'updated_at' => $post->updated_at,
                'content_body' => $post->content_body,
            ],
        ]);
    }

    public function updatePost(Request $request) : \Illuminate\Http\JsonResponse
    {
        $post = Post::find($request->id);

        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        if ($request->has('status')) {
            if (!in_array($request->status, ['DRAFT', 'PUBLISH', 'ARCHIVE'])) {
                return response()->error('Invalid status');
            }
            $post->status = $request->status;
        }

        if ($request->has('publish_date')) {
            $post->publish_date = $request->publish_date;
        }

        if ($request->has('slug')) {
            $post->slug = $request->slug;
        }

        if ($request->has('page_header')) {
            $post->page_header = $request->page_header;
        }

        if ($request->has('page_description')) {
            $post->page_description = $request->page_description;
        }

        if ($request->has('page_og_image')) {
            $post->page_og_image = $request->page_og_image;
        }

        if ($request->has('author')) {
            $post->author = $request->author;
        }

        if ($request->has('content_body')) {
            $contentBody = json_decode($request->content_body, true);
            $post->content_body = $contentBody;
        }

        if ($request->has('tags')) {
            $post->tag_ids = $request->tags;
        }

        if ($request->has('category')) {
            $post->category_id = $request->category;
        }

        $post->save();

        return response()->success('Post updated successful');
    }

    public function deletePost($id) : \Illuminate\Http\JsonResponse
    {
        $post = Post::find($id);

        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $post->delete();

        return response()->success('Post deleted successful');
    }

    public function getPosts(Request $request) : \Illuminate\Http\JsonResponse
    {
        $request->validate([
            'name' => 'string|min:1|max:255',
            'category_id' => 'integer|exists:categories,id',
            'status' => 'string',
        ]);
        $query = Post::query();

        if ($request->filled('name')) {
            $query->whereRaw(
                'LOWER(JSON_EXTRACT(content_body, "$.en.preview_title")) LIKE ?',
                ['%' . strtolower($request->input('name')) . '%']
            )->orWhereRaw(
                'LOWER(JSON_EXTRACT(content_body, "$.ru.preview_title")) LIKE ?',
                ['%' . strtolower($request->input('name')) . '%']
            );
        }

        if ($request->filled('category_id')) {
            $statuses = explode(',', $request->input('category_id'));
            $query->whereIn('category_id', $statuses);
        }

        if ($request->filled('status')) {
            $statuses = explode(',', $request->input('status'));
            $query->whereIn('status', $statuses);
        }

        if ($request->has('page')) {
            $perPage = $request->input('per_page', config('app.per_page', 2));
            $posts = $query->paginate($perPage);
        } else {
            $posts = $query->get();
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags;
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = ['en' => [], 'ru' => []];
            }

            return $post;
        });

        return response()->success('Posts list received successfully', PostResource::collection($posts));
    }

    public function getPostsB2c(Request $request) : \Illuminate\Http\JsonResponse
    {
        $query = Post::query();

        $query->where('status', 'PUBLISH');

        if ($request->has('page')) {
            $perPage = $request->input('per_page', config('app.per_page', 2));
            $posts = $query->paginate($perPage);
        } else {
            $posts = $query->get();
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags; // Аксессор загружает теги

            // Загрузка категории
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = ['en' => [], 'ru' => []];
            }

            return $post;
        });

        return response()->success('Posts list received successfully', $posts);
    }

    public function getPost($id) : \Illuminate\Http\JsonResponse
    {
        $post = Post::find($id);

        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $post->tags = $post->tags;

        $category = Category::find($post->category_id);
        if ($category?->name) {
            $category_name = json_decode($category->name);
            $category_ru[] = ['id' => $category->id, 'name' => $category_name->ru, 'slug' => $category->slug];
            $category_en[] = ['id' => $category->id, 'name' => $category_name->en, 'slug' => $category->slug];

            $post->category = ['en' => $category_en, 'ru' => $category_ru];
        } else {
            $post->category = null;
        }

        return response()->success('Posts list received successful', [
            'posts' => $post,
        ]);
    }

    public function getPostB2c($slug) : \Illuminate\Http\JsonResponse
    {
        $post = Post::query()->where('slug', $slug)->where('status', 'PUBLISH')->first();
        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $post->tags = $post->tags; // Аксессор загружает теги

        $category = Category::find($post->category_id);
        if ($category?->name) {
            $category_name = json_decode($category->name);
            $category_ru[] = ['id' => $category->id, 'name' => $category_name->ru, 'slug' => $category->slug];
            $category_en[] = ['id' => $category->id, 'name' => $category_name->en, 'slug' => $category->slug];

            $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
        } else {
            $post->category = null;
        }

        return response()->success('Posts list received successful', [
            'posts' => $post,
        ]);
    }

    public function getPostsByCategory($category_id) : \Illuminate\Http\JsonResponse
    {
        $category = Category::find($category_id);

        if (!$category) {
            return response()->error('Category not found', [], 422);
        }

        $posts = Post::where('category_id', $category_id)->where('status', 'PUBLISH')->get();

        if ($posts->isEmpty()) {
            return response()->error('Posts not found', [], 422);
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags; // Аксессор загружает теги

            // Загрузка категории
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = null;
            }
            return $post;
        });

        return response()->success('Posts list received successful', [
            'posts' => $posts,
        ]);
    }

    public function getPostsByTag($tag_id) : \Illuminate\Http\JsonResponse
    {
        $posts = Post::whereJsonContains('tag_ids', (int) $tag_id)->where('status', 'PUBLISH')->get();

        if ($posts->isEmpty()) {
            return response()->error('Posts not found', [], 422);
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags; // Аксессор загружает теги

            // Загрузка категории
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = null;
            }
            return $post;
        });

        return response()->success('Posts list received successful', [
            'posts' => $posts,
        ]);
    }
}
