<?php

namespace App\Modules\Blog\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\Blog\Models\Category;
use Illuminate\Http\Request;


class CategoriesController extends Controller
{
    public function getCategories(Request $request) : \Illuminate\Http\JsonResponse
    {
        $categories = $request->has('page')
            ? Category::paginate($request->input('per_page', config('app.per_page', 2)))
            : Category::all();

        $categories->transform(function ($category) {
            $category_json = json_decode($category->name);
            return [
                'id' => $category->id,
                'slug' => $category->slug,
                'name' => [
                    'en' => $category_json->en ?? null,
                    'ru' => $category_json->ru ?? null,
                ],
            ];
        });

        return response()->success('Categories list received successfully', $categories);
    }

    public function createCategory(Request $request) : \Illuminate\Http\JsonResponse
    {
        $request->validate([
            'category_ru' => 'required|string|min:3|max:255',
            'category_en' => 'required|string|min:3|max:255',
            'slug' => 'required|string|min:3|max:255',
        ]);

        $category = new Category();
        $category->name = json_encode(['ru' => $request->category_ru, 'en' => $request->category_en], JSON_UNESCAPED_UNICODE);
        $category->slug = $request->slug;
        $category->save();

        return response()->success('Category created successful');
    }

    public function deleteCategory($id) : \Illuminate\Http\JsonResponse
    {
        $category = Category::find($id);

        if (!$category) {
            return response()->error('Category not found', [], 422);
        }

        $category->delete();

        return response()->success('Category deleted successful');
    }

    public function getCategoriesB2c(Request $request) : \Illuminate\Http\JsonResponse
    {
        $categories = $request->has('page')
            ? Category::paginate($request->input('per_page', config('app.per_page', 2)))
            : Category::all();

        $categories->transform(function ($category) {
            $category_json = json_decode($category->name);
            return [
                'id' => $category->id,
                'slug' => $category->slug,
                'posts_count' => $category->posts()->where('status', 'PUBLISH')->count(),
                'name' => [
                    'en' => $category_json->en ?? null,
                    'ru' => $category_json->ru ?? null,
                ],
            ];
        });

        return response()->success('Categories list received successfully', $categories);
    }
}
